<?php
/**
* 2007-2018 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2018 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

include_once(dirname(__FILE__) . "/autoload.php");

class PayBySquare extends PaymentModule
{
    protected $config_form = false;
    private $stores = array();



    public function __construct()
    {
        $this->name = 'paybysquare';
        $this->tab = 'payments_gateways';
        $this->version = '1.6.0';
        $this->author = 'bysquare.com';
        $this->is_eu_compatible = 1;
    		$this->currencies = true;
    		$this->currencies_mode = 'checkbox';
        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Bank/wire payment (PayBySquare)');
        $this->description = $this->l('QR code generator for PayBySquare module');

        //$this->limited_currencies = array('EUR', "CZK");

        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->l('You have to enable the cURL extension on your server to install this module');
            return false;
        }

        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE', false);
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE', false);

        //include(dirname(__FILE__).'/sql/install.php');

        return parent::install() &&
            $this->registerHook('header') &&
            $this->registerHook('backOfficeHeader') &&
            $this->registerHook('payment') &&
            $this->registerHook('paymentReturn') &&
            $this->registerHook('paymentOptions') &&
            $this->registerHook('actionPaymentConfirmation') &&
            $this->installNewOrderStatuses();
    }

    public function uninstall()
    {
        Configuration::deleteByName('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE');
        Configuration::deleteByName('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE');

        //include(dirname(__FILE__).'/sql/uninstall.php');

        return parent::uninstall() &&
            $this->uninstalNewOrderStatuses();
    }

    protected function _postProcess()
    {
      if (Tools::isSubmit('submitPayBySquareAccount')) {
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_IBAN', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_IBAN'));
        $this->_html = $this->displayConfirmation($this->l('Settings updated'));
      }
      if (Tools::isSubmit('submitPayBySquareModule')) {
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE'));
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE'));
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME'));
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD'));
        Configuration::updateValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY', Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY'));

        //Validate data when turning on PBS payments
        if (Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE') == 1) {
            $this->validateUsernameAndPassword();
        }
        $this->_html .= $this->displayConfirmation($this->l('Settings updated'));
      }
    }

    private function validateUsernameAndPassword()
    {
        $pbs = new PayBySquareAPI();

        if ($pbs->checkBySquareCredentials() == true) {
            //$this->confirmations[] = $this->l("Successful message");
            $this->_html = $this->displayConfirmation($this->l('Paybysquare: Username and Password are OK'));
        } else {
            //$this->errors[] = $this->l('Paybysquare: Username and Password pair does not exists or is disabled.');
            $this->_html = $this->displayError('Paybysquare: Username and Password pair does not exists or is disabled.');
        }
    }

    public static function getAccountsFromConf() {
      $_account = Configuration::get('PRESTAHELP_PAY_BY_SQUARE_IBAN');
      $__accounts = explode(PHP_EOL,$_account);
      $bank_accounts = array();
      foreach($__accounts as $__account) {
        $__account = trim($__account);
        if ($__account === '') {
            continue;
        }

        // IBAN;BIC;Account name (same delimiter as used in BO form)
        $account = explode(";", $__account);
        if (count($account) < 2) {
            continue;
        }

        $bank_accounts[] = array(
            'iban' => trim($account[0]),
            'bic' => trim($account[1])
        );
      }
      return $bank_accounts;
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        $this->_postProcess();

        //Check if shop is selected
        $this->stores = Shop::getContextListShopID();
        if (count($this->stores) > 1) {
            return $this->displayError('Please select shop!');
        }

        $this->context->smarty->assign(array(
            'module_dir' => $this->_path,
           // 'errors'     => $this->errors
        ));
        if(isset($this->_html)) {
          return $this->_html.$this->renderFormAccount().$this->renderForm();
        } else {
          return $this->renderFormAccount().$this->renderForm();
        }
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $fields_form = array(
            'form' => array(
                'legend' => array(
                'title' => $this->l('Settings'),
                'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Live mode'),
                        'name' => 'PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE',
                        'is_bool' => true,
                        'desc' => $this->l('Use this module in live mode'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            )
                        ),
                    ),

                    array(
                      'type' => 'radio',
                      'label' => $this->l('QR code generation'),
                      'name' => 'PRESTAHELP_PAY_BY_SQUARE_LANG_MODE',
                      'hint' => $this->l('Select which category is displayed in the block. The current category is the one the visitor is currently browsing.'),
                      'values' => array(
                        array(
                          'id' => 'slovak',
                          'value' => 0,
                          'label' => $this->l('Generate slovak bank code')
                        ),
                        array(
                          'id' => 'czech',
                          'value' => 1,
                          'label' => $this->l('Generate czech bank code')
                        ),
                        array(
                          'id' => 'basedon',
                          'value' => 2,
                          'label' => $this->l('Generate code based on currency (slovak or czech)')
                        ),
                      )
                    ),
                    array(
                        'col'   => 3,
                        'type'  => 'text',
                        'desc'  => $this->l('Enter paybysquare beneficiary'),
                        'name'  => 'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY',
                        'label' => $this->l('PayBySquare beneficiary name'),
                        'hint'  => $this->l('Name of person or organization receiving money'),

                    ),
                    array(
                        'col'   => 3,
                        'type'  => 'text',
                        'desc'  => $this->l('Enter paybysquare username'),
                        'name'  => 'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME',
                        'label' => $this->l('PayBySquare Username'),
                        'hint'  => $this->l('Your Username for app.bysquare.com service'),
                    ),
                    array(
                        'col'   => 3,
                        'type'  => 'text',
                        'desc'  => $this->l('Enter paybysquare password'),
                        'name'  => 'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD',
                        'label' => $this->l('PayBySquare password'),
                        'hint'  => $this->l('Your Password for app.bysquare.com service'),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitPayBySquareModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );
        return $helper->generateForm(array($fields_form));
    }

    protected function renderFormAccount()
    {
        $fields_form = array(
            'form' => array(
                'legend' => array(
                'title' => $this->l('Account Settings'),
                'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'col'   => 3,
                        'type'  => 'textarea',
                        'desc'  => $this->l('Please enter IBAN, BIC and Account name separated by semicolon. Each line means new account'),
                        'name'  => 'PRESTAHELP_PAY_BY_SQUARE_IBAN',
                        'label' => $this->l('IBAN;BIC;Account name'),
                        'hint'  => $this->l('Please enter IBAN, BIC and Account name separated by semicolon. Each line means new account'),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitPayBySquareAccount';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValuesAccount(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );
        return $helper->generateForm(array($fields_form));
    }



    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return array(
          'PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE')),
          'PRESTAHELP_PAY_BY_SQUARE_LANG_MODE' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')),
          'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_USERNAME')),
          'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_PASSWORD')),
          'PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_PAYBYSQUARE_BENEFICIARY')),
        );
    }
    protected function getConfigFormValuesAccount()
    {
        return array(
          'PRESTAHELP_PAY_BY_SQUARE_IBAN' => Tools::getValue('PRESTAHELP_PAY_BY_SQUARE_IBAN', Configuration::get('PRESTAHELP_PAY_BY_SQUARE_IBAN')),
        );
    }


    /**
    * Add the CSS & JavaScript files you want to be loaded in the BO.
    */
    public function hookBackOfficeHeader()
    {
        if (Tools::getValue('module_name') == $this->name) {
            //$this->context->controller->addJS($this->_path.'views/js/back.js');
            //$this->context->controller->addCSS($this->_path.'views/css/back.css');
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookHeader()
    {
        //$this->context->controller->addJS($this->_path.'/views/js/front.js');
        $this->context->controller->addCSS($this->_path.'/views/css/front.css');
    }

    /**
     * This method is used to render the payment button,
     * Take care if the button should be displayed or not.
     */
    public function hookPayment($params)
    {
        $live_mode = Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE');

        if ($live_mode == false) {
            return false;
        }

        $this->smarty->assign(array('module_dir' => $this->_path,
                                    'cart_id' => $params['cart']->id,
                                    'secure_key' => Context::getContext()->customer->secure_key
                                    ));

        return $this->display(__FILE__, 'views/templates/hook/payment.tpl');
    }

    /*
    * 1.7 compat
    */
    public function hookPaymentOptions($params)
    {
        $live_mode = Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LIVE_MODE');

        if ($live_mode == false) {
            return false;
        }

        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $payment_options = array(
            $this->getOfflinePaymentOption(),
        );

        return $payment_options;
    }

    private function getOfflinePaymentOption()
    {
        $offlineOption = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();

        $order_params = array(
            'cart_id' => $this->context->cart->id,
            'secure_key' => $this->context->customer->secure_key
        );

        $tpl_path = 'module:paybysquare/views/templates/front/payment_infos.tpl';

        $offlineOption->setCallToActionText($this->l('paybysquare'))
                      ->setAction($this->context->link->getModuleLink($this->name, 'confirmation', $order_params, true))
                      ->setAdditionalInformation($this->context->smarty->fetch($tpl_path));
        //->setLogo(Media::getMediaPath(_PS_MODULE_DIR_.$this->name.'/logo.png'));

        return $offlineOption;
    }



    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);
        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * This hook is used to display the order confirmation page.
     */
    public function hookPaymentReturn($params)
    {
        if ($this->active == false) {
            return;
        }

        if (version_compare(Tools::substr(_PS_VERSION_, 0, 3), '1.6', '=')) {
            $order = $params['objOrder'];
            $total_to_pay = $params['total_to_pay'];
            $currencyObj = $params['currencyObj'];
        } else {
            $order = $params['order'];
            $total_to_pay = $params['order']->getOrdersTotalPaid();
            $currencyObj = new Currency((int) $params['order']->id_currency);
        }

        $customer = new Customer($params['cart']->id_customer);

        if ($order->getCurrentOrderState()->id != Configuration::get('PS_OS_ERROR')) {
            $this->smarty->assign('status', 'ok');
        }

        //Params
        //$context = Context::getContext();
        $id_lang = $this->context->language->id;

        $currency = new Currency($order->id_currency);
        //$currency->iso_code

        //Get qr code
        $pbs = new PayBySquareAPI();



        if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==0) {
          if($currency->iso_code=='EUR') {
            $qr_code_info = $pbs->fetchQrcodePngInfo($order->id);
            $qr_code_html = $pbs->outputQrCodeImage($qr_code_info[1]);
            $qractive = 1;
          } else {
            $qractive = 0;
          }
        } elseif(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==1) {
          if($currency->iso_code=="CZK") {
            $qr_code_info = $pbs->fetchQrcodePngInfo($order->id);
            $qr_code_html = $pbs->outputQrCodeImage($qr_code_info[1]);
            $qractive = 1;
          } else {
            $qractive = 0;
          }
        } else {
          if($currency->iso_code=='EUR' || $currency->iso_code=='CZK') {
            $qr_code_info = $pbs->fetchQrcodePngInfo($order->id);
            $qr_code_html = $pbs->outputQrCodeImage($qr_code_info[1]);
            $qractive = 1;
          } else {
            $qractive = 0;
          }
        }


/*
        //ak je ina mena ako eur, alebo czk
        if($currency->iso_code=='EUR') {
          if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==0) {
            $qr_code_info = $pbs->fetchQrcodePngInfo($order->id);
            $qr_code_html = $pbs->outputQrCodeImage($qr_code_info[1]);
            $qractive = 1;
          } else {
            $qractive = 0;
          }
        } elseif($currency->iso_code=="CZK") {
          if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==1) {
            $qr_code_info = $pbs->fetchQrcodePngInfo($order->id);
            $qr_code_html = $pbs->outputQrCodeImage($qr_code_info[1]);
            $qractive = 1;
          } else {
            $qractive = 0;
          }
        } else {
          $qractive = 0;
        }
*/
        $pbsm = new PayBySquareMails();

        if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==0) {
            $bank_details = $pbs->fetchBankDetails("SK");
        } elseif(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==1) {
            $bank_details = $pbs->fetchBankDetails("CZ");
        } else {
          if($currency->iso_code=="EUR") {
            $bank_details = $pbs->fetchBankDetails("SK");
          } elseif ($currency->iso_code=="CZK") {
            $bank_details = $pbs->fetchBankDetails("CZ");
          } else {
            $bank_details = $pbs->fetchBankDetails("SK");
          }
        }


        $this->smarty->assign(array(
            'module_display_name' => $this->displayName,

            'shop_name' => array($this->context->shop->name),
            'bank_accounts' =>  $bank_details,

            'ORDER_DETAILS'   => $pbsm->parseOrderDetails($order->id),
            'ADDRESS_DETAILS' => $pbsm->parseAddressDetails($order->id),

            'order_id' => $order->id,
            'reference' => $order->id,
            'order_date' => Tools::displayDate($order->date_add, false, (int) $id_lang),
            'customer_email' => $customer->email,
            'params' => $params,
            'total' => $this->formatPriceCompat($total_to_pay, $currencyObj),

            'qr_code' => $qr_code_html,
            'qr_code_src' => $qr_code_info[1],
            'qractive' => $qractive,
        ));

        return $this->display(__FILE__, 'views/templates/hook/confirmation.tpl');
    }

    public function hookActionPaymentConfirmation()
    {
    }


    public function installNewOrderStatuses()
    {
        $statuses = array(
           /* 'PRESTAHELP_PAY_BY_SQUARE_NEW_ORDER_STATUS' => array(
                'name' => 'New PayBySquare order',
                'send_email' => false,
                'color' => '#6da1d2',
                'hidden' => false,
                'delivery' => false,
                'logable' => false,
                'invoice' => false,
                'template' => '',
            ),*/
            'PRESTAHELP_PAY_BY_SQUARE_WAYTING_FOR_PAYMENT' => array(
                'name' => 'Awaiting PayBySquare payment',
                'send_email' => false,
                'color' => '#6da1d2',
                'hidden' => false,
                'delivery' => false,
                'logable' => false,
                'invoice' => false,
                'template' => 'pay_by_square_order_confirmation',
            ),
        );

        $success = true;

        foreach ($statuses as $key => $value) {
            $orderState = new OrderState();
            $this->assignLangToArray($orderState->name, $value['name']);
            $orderState->send_email = $value['send_email'];
            $orderState->color = $value['color'];
            $orderState->hidden = $value['hidden'];
            $orderState->delivery = $value['delivery'];
            $orderState->logable = $value['logable'];
            $orderState->invoice = $value['invoice'];
            $this->assignLangToArray($orderState->template, $value['template']);

            if (!$orderState->add()) {
                $success = false;
            }

            Configuration::updateValue($key, (int)$orderState->id);
        }

        return $success;
    }


    public function assignLangToArray(&$object_to_modify, $string)
    {
        foreach (Language::getLanguages() as $language) {
            $object_to_modify[$language['id_lang']] = $string;
        }
    }

    public function uninstalNewOrderStatuses()
    {
        $success = true;

        // Remove "Awaiting PayBySquare payment" order state created on install
        $waitingId = (int) Configuration::get('PRESTAHELP_PAY_BY_SQUARE_WAYTING_FOR_PAYMENT');
        if ($waitingId) {
            $orderState = new OrderState($waitingId);
            if (Validate::isLoadedObject($orderState)) {
                $success = $orderState->delete() && $success;
            }
            Configuration::deleteByName('PRESTAHELP_PAY_BY_SQUARE_WAYTING_FOR_PAYMENT');
        }

        // Clean up legacy config key if it exists from older versions
        if (Configuration::hasKey('PRESTAHELP_PAY_BY_SQUARE_NEW_ORDER_STATUS')) {
            Configuration::deleteByName('PRESTAHELP_PAY_BY_SQUARE_NEW_ORDER_STATUS');
        }

        return $success;
    }

    private function formatPriceCompat($amount, $currency)
    {
        // $currency môže byť Currency objekt alebo ID
        if (!($currency instanceof Currency)) {
            $currency = new Currency((int) $currency);
        }

        // Staršie PS (kde Tools::displayPrice existuje)
        if (method_exists('Tools', 'displayPrice')) {
            return Tools::displayPrice($amount, $currency);
        }

        // PS 9 fallback
        try {
            $formatter = new \PrestaShop\PrestaShop\Adapter\Product\PriceFormatter();
            return $formatter->format($amount, $currency->iso_code);
        } catch (\Throwable $e) {
            return number_format((float)$amount, 2, '.', ' ') . ' ' . $currency->iso_code;
        }
    }

}
