<?php
/**
* 2007-2018 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2018 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

class PayBySquareMails extends Module
{
    private $templates_path;

    public function __construct()
    {
        $this->templates_path =  dirname(__FILE__) . "/../mails/";
    }

    public function sendOrderConfirmationMail($id_cart)
    {
        $cart = new Cart((int)$id_cart);
        $customer = new Customer((int)$cart->id_customer);

        //$order_id = Order::getOrderByCartId((int)$cart->id);
        //$order = new Order($order_id);
        $order_id = (int) Order::getIdByCartId((int) $cart->id);
        if ($order_id <= 0) {
            return false;
        }
        $order = new Order($order_id);
        $currency = new Currency((int) $order->id_currency);

        $pbs = new PayBySquareAPI();
        $qr_code = $pbs->fetchQrcodePngInfo($order_id);
        $qr_code_image = $pbs->outputQrCodeImage($qr_code[1]);

        // Choose bank accounts country based on order currency (CZ -> CZK, otherwise SK)
        $bank_details = $pbs->fetchBankDetails(
            ($currency->iso_code === 'CZK') ? 'CZ' : 'SK'
        );

        //Get Qr code file
        $img_file = Tools::file_get_contents($qr_code[0]);
        $type = 'png';
        $base64 = 'data:image/' . $type . ';base64,' . base64_encode($img_file);

        $extra_vars = array(
            '{shop_url}'        => '',
            '{shop_logo}'       => '',
            '{shop_name}'       => '',
            '{title}'           => 'PAY by square '. $order_id,
            '{REFERENCE}'       => $order->id,
            '{ORDERTOTAL}' => (method_exists('Tools', 'displayPrice'))
                ? $this->formatPrice($order->total_paid_tax_incl, $currency)
                : (new \PrestaShop\PrestaShop\Adapter\Product\PriceFormatter())->format($order->total_paid_tax_incl, $currency->iso_code),

            '{QR_CODE_INLINE}'  => $base64,
            '{QR_CODE}'         => $qr_code_image,
            '{BANK_DETAILS}'    => $this->parseBankAccounts($bank_details),
            '{ORDER_DETAILS}'   => $this->parseOrderDetails($order_id),
            '{ADDRESS_DETAILS}' => $this->parseAddressDetails($order_id),
        );


        $attachments = array();

        $attachments[] = array(
            'content' => $img_file,
            'name'    => 'pay_by_square_' . $order_id . '.png',
            'mime'    => 'image/png'
        );

        $currenc = new Currency($order->id_currency);

        if($currenc->iso_code=="EUR") {
          if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==0) {
            $ttemplate = 'pay_by_square_order_confirmation';
          } else {
            $ttemplate = 'pay_by_square_order_confirmation_no';
          }
        } elseif ($currenc->iso_code=="CZK") {
          if(Configuration::get('PRESTAHELP_PAY_BY_SQUARE_LANG_MODE')==1) {
            $ttemplate = 'pay_by_square_order_confirmation';
          } else {
            $ttemplate = 'pay_by_square_order_confirmation_no';
          }
        } else {
          $ttemplate = 'pay_by_square_order_confirmation_no';
        }

        Mail::Send(
            $order->id_lang,
            $ttemplate,
            'PayBySquare',
            $extra_vars,
            $customer->email,
            $customer->firstname.' '.$customer->lastname,
            "",
            "",
            $attachments,
            null,
            $this->templates_path
        );
    }

    public function parseBankAccounts($bank_accounts)
    {

        $module = Module::getInstanceByName('paybysquare');
        $asdata_smarty = $module->context->smarty;
        $asdata_smarty->caching = false;

         $asdata_smarty->assign(array(
                'bank_accounts' => $bank_accounts,
            ));


        return $asdata_smarty->fetch(dirname(__FILE__) . '/../views/templates/api/bank_information.tpl');
    }

    public function parseOrderDetails($object_id, $type = 'order')
    {
        if ($type == 'order') {
            $order = new Order($object_id);
            $cart = new Cart($order->id_cart);
        } else {
            $cart = new Cart($object_id);
        }

        $order_date_parsed = '';
        if ($type === 'order') {
            $order_date_parsed = Tools::displayDate(
                $order->date_add,
                true,                 // alebo false (full vs lite formát)
                (int) $order->id_lang // alebo (int) Configuration::get('PS_LANG_DEFAULT')
            );
        }

        $cart_sumamry = $cart->getSummaryDetails();
        $product_detail = $cart->getProducts();
        $currency = new Currency($cart->id_currency);

        foreach ($product_detail as &$p) {
            $p['total_wt_formatted'] = $this->formatPrice((float)$p['total_wt'], $currency);
        }
        unset($p);

        $module = Module::getInstanceByName('paybysquare');
        $asdata_smarty = $module->context->smarty;
        $asdata_smarty->caching = false;

        $only_products = $this->formatPrice($cart->getordertotal(true, Cart::ONLY_PRODUCTS), $currency);

        $shipping_wf = $cart->getordertotal(true, Cart::ONLY_SHIPPING);
        if ($shipping_wf == 0) {
            $shipping = '0';
        } else {
            $shipping =  $this->formatPrice($shipping_wf, $currency);
        }

        if ($cart_sumamry['total_discounts'] == 0) {
            $total_discounts = '0';
        } else {
            $total_discounts = $this->formatPrice($cart_sumamry['total_discounts'], $currency);
        }

        $total_order = $this->formatPrice($cart->getordertotal(true), $currency);

        $asdata_smarty->assign(
            array(
                'type' => $type,
                'currency' => $currency,
                'object_id' => $object_id,
                'only_products' => $only_products,
                'shipping' => $shipping,
                'total_discounts' => $total_discounts,
                'total_order' => $total_order,
                'order_date_parsed' => $order_date_parsed,
                'product_detail' => $product_detail,
            )
        );


        return $asdata_smarty->fetch(dirname(__FILE__) . '/../views/templates/api/cart_summary.tpl');
    }

    public function parseAddressDetails($object_id, $type = 'order')
    {
        if ($type == 'order') {
            $object = new Order($object_id);
        } else {
            $object = new Cart($object_id);
        }

        $customer = new Customer($object->id_customer);

        $delivery_address = new Address($object->id_address_delivery);
        $invoice_address = new Address($object->id_address_invoice);

        $module = Module::getInstanceByName('paybysquare');
        $asdata_smarty = $module->context->smarty;
        $asdata_smarty->caching = false;


        $asdata_smarty->assign(
            array(
                'delivery_address' => (array) $delivery_address,
                'invoice_address' =>  (array) $invoice_address,
                'customer' => (array) $customer
            )
        );


        return $asdata_smarty->fetch(dirname(__FILE__) . '/../views/templates/api/address_summary.tpl');
    }

    private function formatPrice($amount, Currency $currency)
    {
        if (method_exists('Tools', 'displayPrice')) {
            return Tools::displayPrice($amount, $currency);
        }

        try {
            $formatter = new \PrestaShop\PrestaShop\Adapter\Product\PriceFormatter();
            return $formatter->format($amount, $currency->iso_code);
        } catch (\Throwable $e) {
            return number_format((float)$amount, 2, '.', ' ') . ' ' . $currency->iso_code;
        }
    }

}
